import React, { Component } from 'react';
import PropTypes from 'prop-types';
import { Link } from 'react-router-dom';
import Button from '../Button/Button';
import Form, { InputRow, FormHint, FormRow } from '../Form/Form';
import Icon from '../Icon/Icon';
import Divider from '../Divider/Divider';
import { register, signInGoogle } from '../../actions/auth';

class FormRegister extends Component {
  static propTypes = {
    dispatch: PropTypes.func.isRequired,
    isSubmitting: PropTypes.bool.isRequired,
    onSubmit: PropTypes.func,
    errors: PropTypes.shape({
      name: PropTypes.string,
      email: PropTypes.string,
      password: PropTypes.string,
    }),
  };

  static defaultProps = {
    onSubmit: null,
    errors: {},
  };

  state = {
    email: '',
    password: '',
    name: '',
  };

  onChangeField = key => e => {
    const { value } = e.target;
    this.setState({ [key]: value });
  };

  onSubmit = e => {
    const { dispatch, onSubmit, isSubmitting } = this.props;
    const { email, password, name } = this.state;

    e.preventDefault();
    if (isSubmitting) {
      return;
    }

    dispatch(register(email, password, name));
    if (typeof onSubmit === 'function') {
      onSubmit(email, password, name);
    }
  };

  render() {
    const { name, email, password } = this.state;
    const { errors, isSubmitting, dispatch } = this.props;

    return (
      <Form className="form_auth" onSubmit={e => this.onSubmit(e)}>
        <FormRow>
          <Button
            type="button"
            styleType="google"
            onClick={() => dispatch(signInGoogle())}
            disabled={isSubmitting}
            isFullWidth
          >
            Continue com o Google <Icon name="arrow" />
          </Button>
        </FormRow>
        <Divider>Ou registre -se no seu e -mail </Divider>
        <InputRow
          name="name"
          label="Seu nome"
          value={name}
          error={errors.name}
          onChange={this.onChangeField('name')}
          required
        />
        <InputRow
          type="email"
          name="email"
          label="Seu email"
          value={email}
          error={errors.email}
          onChange={this.onChangeField('email')}
          required
        />
        <InputRow
          type="password"
          name="password"
          label="Senha"
          placeholder="Senha (pelo menos 6 caracteres)"
          value={password}
          error={errors.password}
          onChange={this.onChangeField('password')}
          minLength="6"
          required
        />
        <FormRow type="submit">
          <Button type="submit" styleType="primary" disabled={isSubmitting} isFullWidth>
            Crie a sua conta aqui
          </Button>
        </FormRow>
        <FormRow className="text_align_center">
          <FormHint>
            Ao assinar ou criar uma conta, você concorda com nosso{' '}
            <Link to="/terms-of-services" target="_blank">
              Termos{'\u00A0'}do{'\u00A0'}Serviço
            </Link>{' '}
            e{' '}
            <Link to="/cookie-policy" target="_blank">
             Declaração de privacidade
            </Link>
          </FormHint>
        </FormRow>
      </Form>
    );
  }
}

export default FormRegister;
